#!/bin/bash
##############################################################################
# TPCH Raw Data Generation 
##############################################################################
set -eu
configPath=$(dirname $(dirname "$PWD"))/conn-params.conf
source ../params.conf

if [ $# -lt 1 ]; then
	echo "[ERROR] Insufficient # of params"
	echo "USAGE: `dirname $0`/$0 <scaleFactor> "
	exit 127
fi

scaleFactor=$1

### Check for cores per worker
if [ -z $threadsPerWorker ]; then
	echo "[ERROR] Cores per worker was not configured"
	exit 127
fi

### Check for remote generating directory
if [ -z $remoteGenDir ]; then
	echo "[ERROR] Remote generating directory was not configured"
	exit 127
fi

### Check for  target path
if [ -z $targetPath ]; then
	echo "[ERROR] target path was not configured"
	exit 127
fi


### Check for workload -- a file with one row for a tpc-h table, each row with format <tblCode> <tblName> <#Chunks>
## Assume that all the nodes are uniformi. If no existing workload file is found, will generate one now -
if [ $scaleFactor -ge 1000 ]; then
	workload=1000
elif [ $scaleFactor -ge 100 ]; then
	workload=100
elif [ $scaleFactor -ge 10 ]; then
	workload=10
elif [ $scaleFactor -ge 1 ]; then
	workload=1
else
	echo "Illegal scaleFactor: $scaleFactor" >&2
	exit 1
fi

workFile=../workloads/tpch.workload.${workload}.lst

echo $workFile
#Clear existing workloads
rm -rf writeData.sh
rm -rf ${targetPath}/SF${scaleFactor}

###
#Creating Root Directory (if not existent)
echo "[INFO] Creating Root Directory ${targetPath}/SF${scaleFactor} (if not existent)"
mkdir -p ${targetPath}/SF${scaleFactor}

### Init Workloads
fileName=writeData.sh
echo "#!/bin/bash" >> $fileName
echo "  "  >> $fileName
echo "workDir=\`dirname \$0\`"  >> $fileName
echo "cd \$workDir"  >> $fileName
echo "export DSS_PATH=$remoteGenDir" >> $fileName
echo "export configPath=$configPath" >> $fileName
echo "source ../params.conf"  >> $fileName
echo "  "  >> $fileName

### Generate Workloads
while read line; do
	if [ x"$line" == "x" ]; then
		continue;
	fi
	params=( $line )
	#Extracting Parameters
	tblCode=${params[0]}
	tblName=${params[1]}
	totalParts=${params[2]}
	tblStarts=${params[3]}
	tblEnds=${params[4]}

	echo "====$tblName==="
	mkdir -p ${targetPath}/SF${scaleFactor}/$tblName
	# Assigning workload in round-robin fashion
	partsDone=$tblStarts
	while [ $partsDone -le $tblEnds ]; do
		echo "$remoteGenDir/localDbGen.sh $scaleFactor $tblCode $partsDone $totalParts ${targetPath}/SF${scaleFactor}/${tblName}/ &" >> writeData.sh
		echo "$remoteGenDir/wait4process.sh dbgen \$threadsPerWorker " >> writeData.sh
		echo "   " >> writeData.sh

		let partsDone=1+$partsDone
	done
done <$workFile;

echo "wait" >> writeData.sh

echo "[INFO] Started Generation @ "`date +%H:%M:%S`
startTime=`date +%s`

### Distribute & Execute workloads

chmod 755 writeData.sh
sh writeData.sh &


### Waiting for completion
echo "[INFO] Waiting until completion..."
wait
endTime=`date +%s`
echo "[INFO] Completed Generation @ "`date +%H:%M:%S`
echo "[INFO] Generated and loaded SF"${scaleFactor}" in "`echo $endTime - $startTime |bc`" sec"
